﻿using System;
using System.Collections;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using Inet.Viewer.Data;

namespace Inet.Viewer.WinForms.Prompt
{
    /// <summary>
    /// Responsible for creating the necessary panel for the given PromptData object
    /// </summary>
    public class PromptPanelBuilder
    {
        /// <summary>
        /// Creates the Control for adding to the right-hand panel of the prompt dialog
        /// </summary>
        /// <param name="p">PromptData object for which to create the prompt panel</param>
        /// <param name="validationDelegate">Validation Delegate when validation is complete</param>
        /// <returns>Created prompt panel. Will return null if prompt type is unknown.</returns>
        public Control CreateControlFor(PromptData p, ValidationDelegate validationDelegate)
        {
            if (p.MultipleAllowed)
            {
                PromptControl c1 = null, c2 = null;
                if (p.DiscreteAllowed)
                {
                    c1 = CreateControlFor(p, validationDelegate, false);
                }
                if (p.RangeAllowed)
                {
                    c2 = CreateControlFor(p, validationDelegate, true);
                }
                return new MultiPromptField(p, c1, c2);
            }
            else
            {
                return CreateControlFor(p, validationDelegate, p.RangeAllowed);
            }
        }
        /// <summary>
        /// creates the control for the given PromptData object with the given parameters
        /// </summary>
        /// <param name="p">PromptData object to create the Control for</param>
        /// <param name="validationDelegate">validation delegate for validating chosen prompt values</param>
        /// <param name="isRange">whether to force the panel to be range</param>
        /// <returns></returns>
        private PromptControl CreateControlFor(PromptData p, ValidationDelegate validationDelegate, bool isRange)
        {
            bool hasDefaults = p.DefaultValues != null && p.DefaultValues.Count > 0;
            bool hasRangeDefaults = p.DefaultValues != null && p.DefaultValues.Cast<PromptValue>().Any(e => e is RangePromptValue);
            bool hasSingleDefaults = p.DefaultValues != null && p.DefaultValues.Cast<PromptValue>().Any(e => !(e is RangePromptValue));
            switch (p.Type)
            {
                case PromptData.String:
                case PromptData.Number:
                case PromptData.Currency:
                    if (isRange)
                    {
                        if (hasDefaults)
                        {
                            return new RangePromptFieldWithDefaultValues(p);
                        }
                        else
                        {
                            return new RangePromptField(p);
                        }
                    }
                    else
                    {
                        if (hasDefaults)
                        {
                            return new SinglePromptFieldWithDefaultValues(p);
                        }
                        else
                        {
                            return new SinglePromptField(p,validationDelegate);
                        }
                    }
                case PromptData.Boolean:
                    return new SingleBooleanPromptField(p);
                case PromptData.Date:
                case PromptData.Datetime:
                    if (isRange)
                    {
                        return new DateRangePromptField(p);
                    }
                    else if (hasDefaults)
                    {
                        return new DefaultValueField(p, new DatePromptField(p, true), false);
                    }
                    else
                    {
                        return new DatePromptField(p, false);
                    }
                case PromptData.Time:
                    PromptControl ctrl;
                    if (isRange)
                    {
                        ctrl = new TimeRangePromptField(p, validationDelegate, hasDefaults);
                        if (hasRangeDefaults)
                        {
                            ctrl = new DefaultValueField(p, ctrl, true);     
                        }
                    }
                    else
                    {
                        ctrl =  new TimePromptField(p, validationDelegate, hasDefaults);
                        if (hasSingleDefaults)
                        {
                            ctrl = new DefaultValueField(p, ctrl, false);
                        }
                    }
                    return ctrl;     
                default:
                    return null;
            }
        }
    }
}
